/*  Suche im vorgegebenen Zeitfenster nach freien Zeitslots auf der gegebenen Ressource,
    wo ein AG mit gegebener Auslastung und gegebener Zeitdauer eingeplant werden könnte.

    Mit dem Parameter "_allow_fragmentation" kann Fragmentierung erlaubt werden. Eine Fragementierung liegt nur dann vor, wenn
    innerhalb der Abarbeitung eines AG die Bearbeitung eines anderen AG dazwischengeschoben wird. Andere Unterbrechungsgründe
    der Abarbeitung eines AG (Maschinenpausenzeiten, etc.) zählen nicht als Fragmentierung. Folgende drei Modi sind möglich:
    + 'no'    - Keine Fragmentierung erlauben. Komplette Laufzeit muss ohne Fragmentierung terminerit werden können, sonst wird
                die nächste Lücke nach der aufgtretenen Fragmentierung probiert.
    + 'until' - Es wird die erste gefundene Lücke im Zeitfenster genutzt, um den AG zu terminieren. Je nachdem was eher
                eintritt, wird die gesamte Laufzeit des AG eingeplant oder der AG nur bis zur ersten Fragmentierung terminiert.
                Die restliche fehlende Laufzeit muss zusätzlich blockiert werden. Es wird versucht mindestens
                einen Timeline-Eintrag für den AG zu erstellen.
    + 'yes'   - Fragmentierung bei der Terminierung erlauben, ab der Fragmentierung werden nur noch Blockierungen
                (task.blocktime) gebildet.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__timeslots__search', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__timeslots__search(
      _resource_id          integer,
      _time_required        numeric,
      _load_required        numeric,
      _timeframe_start      timestamp,
      _timeframe_end        timestamp,
      _direction            varchar = 'forward',
      _scenario             varchar = null,         -- TODO AXS: Unbenutzt! Kann das weg?
      _allow_fragmentation  varchar = 'no',         -- Fragmeniering erlauben: 'no' - keine Fragmentierung erlauben (Standard); 'until' - Terminierung bis zur ersten Fragmentierung durchführen; 'yes' - Fragmentierung erlauben, ab Fragmentierung werden nur noch Blockierungen gebildet.
      _blocktime_refcursor  refcursor = null,       -- Liste von von außen übergebene Timeline-Einträge
      _checkBlockedTimes    bool = true,            -- Bei 'false' DLZ-Terminierung: Ignoriert Blockierungen auf der Ressource durch andere Task*-Einträge. Es werden nur Off-Times beachtet.
      _factor_tm_ta         numeric = 1.0,          -- Personalzeit-zu-Abarbeitungszeit-Faktor = a2_tm / a2_ta
      _isBuffer             bool = false,           -- Werden hier Task-Blöcke für einen Puffer gesucht (TSystem.ENUM_ContainsValue(ab_stat, 'BUFFER'))
      _loglevel             int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS TABLE (                       -- Entspricht dem Datentyp "scheduling.resource_timeslotset"
      -- Hauptressource
      slotStartDate       timestamp,                                -- Startzeit des aktuellen Slots auf der Hauptressource
      slotEndDate         timestamp,                                -- Endzeit des aktuellen aktuellen Slots auf der Hauptressource
      slotType            scheduling.resource_timeline_blocktype,   -- Typ des aktuellen Slots auf der Hauptressource
      slotFactor          numeric,                                  -- Korrekturfaktor des aktuellen Slots auf der Hauptressource
      usage               numeric,                                  -- Belastung des aktuellen Slots auf der Hauptressource
      -- ggf. verknüpfte Kopfkostenstelle
      top_resource_id     integer,                                  -- Resourcen-ID der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
      top_slotStartDate   timestamp,                                -- Startzeit des aktuellen Slots auf der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
      top_slotEndDate     timestamp,                                -- Startzeit des aktuellen Slots auf der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
      top_slotType        scheduling.resource_timeline_blocktype,   -- Typ des aktuellen Slots auf der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
      top_slotFactor      numeric,                                  -- Korrekturfaktor des aktuellen Slots der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
      top_usage           numeric                                   -- Belastung des aktuellen Slots der Kopfkostenstelle (bleibt leer, wenn es keine KKS gibt)
  ) AS $$

  DECLARE

      _shorthand            varchar;
      _prefix               varchar;


      _rowHR                  record; -- Belegungsteiten auf Hauptressource
      _rowKKS                 record; -- Belegungszeiten in Kombination mit Kopfkostenstelle

      _row_factor             record; -- Korrekturfaktor des aktuellen Slots auf der Hauptressource.

      _currentSlot            scheduling.resource_timeslotset;    -- Daten eines einzelnen Slots.
      _currentSet             scheduling.resource_timeslotset[];  -- Liste der Slots.
      _oldSlot                scheduling.resource_timeslotset;
      _newSlot                scheduling.resource_timeslotset;

      _useableSlotSize        numeric = 0;  -- Maximale Slotgröße der aktuellen Lücke
      _slotSizeUsed           numeric = 0;  -- Größe des tatsächlich genutzten Stlots in der Lücke. Ist kleiner gleich der Größe der Lücke.
      _time_performed         numeric = 0;  -- Bis schon verplante Arbeitszeit der zu verplanenden Arbeitszeit.

      _top_resource_id        integer;    -- resource_id der Kopfkostenstelle
      _top_load_used          numeric;    -- Belastung der Kopfkostenstelle durch die bisher an diesem Tag einterminierten Slots.
      _top_load_to_use        numeric;    -- Belastung der Kopfkostenstelle für aktuell einzuterminierenden Slot.
      _top_overloaded         bool;       -- Kapazität der Kopfkostenstelle ist überlastet.

      _top_timeframe_start    timestamp;
      _top_timeframe_end      timestamp;

      _top_ksb_id             integer;
      _top_shifttimes         time[2];
      _top_work_time_day      numeric;
      _top_ksb_ks_ba          numeric;

      _fragmentation_start    timestamp;
      _slotType               scheduling.resource_timeline_blocktype;
      _time_grid              interval := TSystem.Settings__GetInteger( 'scheduling.time_grid', 60 ) * interval '1 second';

  BEGIN
      -- Debug
      IF (-- Exception Case
             _time_required IS NULL
          OR _time_required <= 0
          --Logleven
          OR _loglevel >= 4
          )
      THEN
        _shorthand := k.ksb_ks_shorthand FROM scheduling.resource__translate__resource_id__to__ksvba__shorthand( _resource_id ) k;
        _prefix    := format( 'resource_timeline__timeslots__search rid %L, shorthand %L -', _resource_id, _shorthand );
      END IF;

      IF _loglevel >= 4 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__timeslots__search( _resource_id => %L, _time_required => %L, _load_required => %L, _timeframe_start => %L, _timeframe_end => %L, _direction => %L, _scenario => %L, _allow_fragmentation => %L, _blocktime_refcursor => %L, _checkBlockedTimes => %L, _factor_tm_ta => %L, _isBuffer => %L, _loglevel => %L )',
                                                                                              _resource_id      , round( _time_required, 4 ), round( _load_required, 4 ), _timeframe_start, _timeframe_end, _direction      , _scenario      , _allow_fragmentation      , _blocktime_refcursor      , _checkBlockedTimes      , round( _factor_tm_ta, 4 ), _isBuffer, _loglevel
                                  )
          ;
      END IF;

      IF (
            _time_required IS NULL
         OR _time_required <= 0
      ) THEN
          -- RAISE EXCEPTION '% illegal _time_required: %', _prefix, _time_required;
          -- 0 Zeit wird gesucht (AG mit keiner Vorgabe aber auch kein Bedarfsarbeitsgang) => Startpunkt zurückgeben.
          IF _direction = 'forward' THEN
            slotStartDate := _timeframe_start;
            slotEndDate   := _timeframe_start;
          ELSE
            slotStartDate := _timeframe_end;
            slotEndDate   := _timeframe_end;
          END IF;
          slotFactor := 1;
          usage := 1;
      END IF;

      _currentSet := array[]::scheduling.resource_timeslotset[];

      -- Ressourcen-ID der Kopfkostenstelle ermittlen.
      _top_resource_id := scheduling.resource__top_resource_id__get( _resource_id );

      -- Den Standard-Typ bestimmen (Task oder Puffer).
      IF _isBuffer IS true THEN
          _slotType := 'task.buffer';
      ELSE
          _slotType := 'task';
      END IF;

      -- Kapazitätsstack-Algorithmus
      -- Beschreibung der Terminierungsalgorithmen: https://redmine.prodat-sql.de/projects/terminierung-2021/wiki/Terminierungsalgorithmus#Kopfkostenstellen
      IF TSystem.Settings__GetBool( 'scheduling.algorithm_is_kapa_stack' ) THEN

          -- Durchlaufe alle Belegungszeiten auf der gegebenen Ressource (Hauptressource) im gegebenen Zeitfenster.
          << timeSlotLoop >>
          FOR _rowHR IN

              SELECT
                  a.ids,
                  a.load,
                  a.date_start,
                  a.date_end,
                  a.types,
                  a.next_start,
                  a.next_end,
                  a.prev_start,
                  a.prev_end,
                  a.next_types,
                  a.next_load,
                  a.prev_types,
                  a.prev_load
              -- Belegungszeiten der Hauptresource
              FROM scheduling.resource_timeline__select(
                        _resource_id        => _resource_id,
                        _target_load        => _load_required,
                        _scenario           => _scenario,
                        _timeframe_start    => _timeframe_start,
                        _timeframe_end      => _timeframe_end,
                        _blocktime_cursor   => _blocktime_refcursor,
                        _checkBlockedTimes  => _checkBlockedTimes,
                        _loglevel           => _loglevel
                  ) a
              ORDER BY
                  CASE WHEN _direction = 'forward' THEN a.date_end END ASC,
                  CASE WHEN _direction <> 'forward' THEN a.date_start END DESC

          LOOP
              -- Fragemtierung hat (noch) NICHT begonnen oder es handelt sich um Rückwärtsterminierung, ...
              IF ( _fragmentation_start IS NULL ) OR ( _direction <> 'forward' ) THEN
                  -- ... dann Slot als normalen "Task" festlegen.
                  _currentSlot.slotType := _slotType;
                  _currentSlot.usage    := _load_required;
              -- Fragemtierung hat (schon) begonnen und Terminierungsrichtund ist vorwärts, ...
              ELSE
                  -- ... dann Slot als Blockierung festlegen.
                  _currentSlot.slotType := 'task.blocktime';
                  _currentSlot.usage    := ifthen( _load_required = 0.0, 1.0, _load_required );
              END IF;

              -- Slotgrenzen bestimmen. Erstmal die Grenzen der Lücke verwenden.
              IF ( _direction = 'forward' ) THEN
                  IF ( _rowHR.next_start IS NULL ) THEN
                    CONTINUE timeSlotLoop;
                  END IF;
                  _currentSlot.slotStartDate  := _rowHR.date_end;
                  _currentSlot.slotEndDate    := _rowHR.next_start;
              ELSE -- backward
                  IF ( _rowHR.prev_end IS NULL ) THEN
                    CONTINUE timeSlotLoop;
                  END IF;
                  _currentSlot.slotStartDate  := _rowHR.prev_end;
                  _currentSlot.slotEndDate    := _rowHR.date_start;
              END IF;

              -- Debugausgabe zu Slot
              IF _loglevel >= 5 THEN
                  RAISE NOTICE '% slot: %: % ~> %,', _prefix, _direction, _currentSlot.slotStartDate, _currentSlot.slotEndDate;
              END IF;

              -- Korrekturfaktor für die aktuellen Zeitgrenzen holen.
              SELECT *
              INTO _row_factor
              FROM scheduling.resource_timeline__timeslots_ta_kf__calc(
                       _timeframe_start  => _currentSlot.slotStartDate,
                       _timeframe_end    => _currentSlot.slotEndDate,
                       _resource_id      => _resource_id
                   );
              _currentSlot.slotStartDate  := _row_factor.slotStartDate;
              _currentSlot.slotEndDate    := _row_factor.slotEndDate;
              _currentSlot.ta_kf          := _row_factor.ta_kf;
              _useableSlotSize := ceil( extract ( epoch FROM _row_factor.slotEndDate - _row_factor.slotStartDate )::numeric );  -- Wir wollen nur auf ganze Sekunden aufgerundete Intervalgrößen, da es ansosnsten zu komischen Rundungseffekten kommen kann. Daher hier die Verwendung der Funktion "ceil".

              -- Die Slotgrenzen ggf. anpassen: Wenn die noch zu planende Arbeitszeit größer gleich die Lücke ist, dann belege die weiterhin volle aktuelle Lücke.
              IF ( _time_performed + ceil( _useableSlotSize * _currentSlot.ta_kf ) <= _time_required ) THEN
                  _slotSizeUsed := _useableSlotSize;
              ELSE -- Die noch zu planende Arbeitszeit passt mehr als die aktuelle Lücke.
                  _slotSizeUsed := ceil( ( _time_required - _time_performed ) / _currentSlot.ta_kf );                           -- Wir wollen nur auf ganze Sekunden aufgerundete Intervalgrößen, da es ansosnsten zu komischen Rundungseffekten kommen kann. Daher hier die Verwendung der Funktion "ceil".
                  IF _direction = 'forward' THEN
                      _currentSlot.slotEndDate := _currentSlot.slotStartDate + _slotSizeUsed * interval '1s';
                  ELSE -- rückwärts
                      _currentSlot.slotStartDate := _currentSlot.slotEndDate - _slotSizeUsed * interval '1s';
                  END IF;
              END IF;

              -- Varibale für die Überlastung der Kopfkostenstelle für diesen Schleifendurchlauf vorbelegen.
              _top_overloaded := false;

              -- Nur wenn eine Kopfkostenstelle vorhanden ist.
              IF _top_resource_id IS NOT null THEN
                  -- Belastung der Kopfkostenstelle durch die bisher an diesem Tag einterminierten Slots bestimmen.
                  SELECT sum( ti_usage )
                    INTO _top_load_used
                    FROM scheduling.resource_timeline
                   WHERE ti_resource_id = _top_resource_id
                     AND ti_date_start  = _currentSlot.slotStartDate::date;
                  -- Belastung der Kopfkostenstelle für den aktuell einzuplanenden Slot ermittlen.
                  _top_load_to_use := scheduling.resource_timeline__timeslots_top_usage__calc(
                                          _top_resource_id  => _top_resource_id
                                        , _date             => CASE WHEN _direction = 'forward' THEN _currentSlot.slotStartDate ELSE _currentSlot.slotEndDate END::date
                                        , _slotSizeUsed     => ( ceil( _slotSizeUsed * _currentSlot.ta_kf ) ) / ( 60 * 60 )
                                        , _factor_tm_ta     => _factor_tm_ta
                                        , _loglevel         => _loglevel
                                      );
                  -- Prüfen, ob die Kopfkostenstelle durch das Einterminieren des aktuellen Slots überlastet wird. (Bisherige + hinzukommende Belastung größer 1.)
                  IF ( _top_load_used + _top_load_to_use > 1 ) THEN
                      _top_overloaded := true;
                  END IF;
                  -- Debugausgabe zur Überlastung der KKS.
                  IF _loglevel >= 6 THEN
                      RAISE NOTICE '%   >> [check for overload] top_load_used: %, top_load_to_use: %, top_overloaded: %;', _prefix, round( _top_load_used, 4 ), round( _top_load_to_use, 4 ), _top_overloaded;
                  END IF;
              END IF;

              -- Geplante Arbeistzeit aktualisieren.
              _time_performed := _time_performed + ceil( _slotSizeUsed * _currentSlot.ta_kf );

              -- Fragmentierung ist verboten oder hat noch nicht gestartet oder Terminierungsrichtung ist rückwärts, ...
              -- Bei erlaubter Fragmentierung und Terminierungsrichtung rückwärts IMMER auf Fragmentierung prüfen, auch wenn schon Fragmentierung erkannt wurde, um letztendlich den frühesten Fragmentierungszeitpunkt zu finden, aber nur, wenn vorher schon ein freier Slot belegt wurde.
              IF ( ( _allow_fragmentation <> 'yes' ) OR ( _fragmentation_start IS NULL ) OR ( _direction <> 'forward' ) ) AND ( _time_performed > 0 ) THEN
                  -- ... dann auf Fragmentierung prüfen:
                  -- Debugausgabe zur Prüfung auf Fragmentierung
                  IF _loglevel >= 6 THEN
                      IF _direction = 'forward' THEN
                          RAISE NOTICE '%   >> [check for fragmentation] next_types: %, next_load: %, load_required: %;', _prefix, _rowHR.next_types, _rowHR.next_load, _load_required;
                      ELSE -- backward
                          RAISE NOTICE '%   >> [check for fragmentation] prev_types: %, prev_load: %, load_required: %;', _prefix, _rowHR.prev_types, _rowHR.prev_load, _load_required;
                      END IF;
                  END IF;

                  -- Prüfung, ob die nächste Belegung eine Abarbeitung eines anderen AG ("task") ist, aber nur, wenn noch zu verplanende Arbeitszeit für den aktuellen AG übrig ist.
                  -- Andere Belegungsgründe als "task", "task.blocktype" bzw. "task.buffer" zählen nicht als Fragmentierung.
                  -- Ein Task-Block zählt nur, wenn die gesamtmögliche Belegung durch den hinzukommenden aktuellen AG überschritten werden würde.
                  -- Bei den Kopfkostenstellen darf es Fragmentierung geben.
                  -- Aber die Kopfkostenstellen dürfen durch Überlastete Tage keine Lücken in der Terminierung auf der Hauptressource verursachen.
                  IF (
                        -- Vorwärtsterminierung, Hauptressource
                        (
                              _direction = 'forward'
                          AND (
                                    _rowHR.next_types && array[ 'task', 'task.blocktime' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task'
                                OR _rowHR.next_types && array[ 'task', 'task.blocktime', 'task.buffer' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task.buffer'
                              )
                          AND _rowHR.next_load + _load_required > 1
                          AND _time_performed < _time_required
                        )
                      OR
                        -- Rückwärtsterminierung, Hauptressource
                        (
                              _direction <> 'forward'
                          AND (
                                    _rowHR.prev_types && array[ 'task', 'task.blocktime' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task'
                                OR _rowHR.prev_types && array[ 'task', 'task.blocktime', 'task.buffer' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task.buffer'
                              )
                          AND _rowHR.prev_load + _load_required > 1
                          AND _time_performed < _time_required
                        )
                      OR
                        -- Kapazität der Kopfkostenstelle ist überlastet.
                        -- Das führt zu einer Lücke auf der Hauptressource, welche wiederum zu Fragmentierung führen kann.
                        -- In diese Lücke könnte ein Task-Block eines anderen kürzeren AGs einterminiert werden, dann haben wir Fragmentierung.
                        -- Interessiert nur wenn KEINE DLZ-Terminierung. Bei DLZ-Terminierung darf auch KKS überlastet werden.
                        (
                          _top_overloaded AND _checkBlockedTimes
                        )
                  ) THEN
                      -- Fragmentierung ist verboten, ...
                      IF ( _allow_fragmentation = 'no' ) THEN
                          -- ... dann wird das Ergebnis der aktuellen Suche nach freien Blöcken verworfen und es wird ab dem nächsten Belegungsblock wieder nach freien Lücken gesucht.
                          IF _loglevel >= 5 THEN
                              RAISE NOTICE '%   >> [reset search] due to fragmentation', _prefix;
                          END IF;

                          -- Vergesse bisherige Blöcke.
                          _time_performed := 0;
                          _currentSet := array[]::scheduling.resource_timeslotset[];
                          -- Springe zum nächsten Schleifendurchlauf.
                          CONTINUE timeSlotLoop;
                      -- Fragmentierung ist erlaubt, ...
                      ELSE
                          -- ... dann merken, das ab nächstem Block die Fragmentierung begonnen hat, aber nur, wenn noch keine Fragmentierung gefunden wurde.
                          IF ( _fragmentation_start IS null ) OR ( _direction <> 'forward' ) THEN
                              IF _direction = 'forward' THEN
                                  _fragmentation_start := _currentSlot.slotEndDate;
                              ELSE
                                  _fragmentation_start := _currentSlot.slotstartDate;
                              END IF;
                              IF _loglevel >= 5 THEN
                                  RAISE NOTICE '%   >> [fragmentation found] fragmentation_start: %', _prefix, _fragmentation_start;
                              END IF;
                          END IF;
                      END IF;
                  END IF;
              END IF;

              -- Prüfen, ob der aktuelle Slot auf der Hauptressource die Länge 0 hat, ...
              IF ( _useableSlotSize = 0 ) THEN
                  IF _loglevel >= 5 THEN
                      RAISE NOTICE '%   >> [skip block] due to zero slot size', _prefix;
                  END IF;
                  -- ... dann zum nächsten Schleifendurchlauf springen.
                  CONTINUE timeSlotLoop;
              END IF;

              -- Prüfen, ob der aktuelle Slot auf der Hauptressource einen Korrekturfaktor von 0 hat, ...
              -- Diese blöcke erledigen keine Arbeit ( _time_performed = 0 ). Kann z.B. an Feiertagen auftreten.
              IF ( _currentSlot.ta_kf = 0 ) THEN
                  IF _loglevel >= 5 THEN
                      RAISE NOTICE '%   >> [skip block] due to zero ta_kf', _prefix;
                  END IF;
                  -- ... dann zum nächsten Schleifendurchlauf springen.
                  CONTINUE timeSlotLoop;
              END IF;

              -- Kopfkostenstelle existiert und Fragmentierung hat (noch) NICHT begonnen oder Fragmentierungsrichtung ist rückwärts, ...
              IF ( _top_resource_id IS NOT null ) AND ( ( _currentSlot.slotStartDate < coalesce( _fragmentation_start, 'infinity'::timestamp ) ) OR ( _direction <> 'forward' ) ) THEN
                  -- Die Arbeitszeit der Kopfkostenstelle ermitteln.
                  SELECT ksb_id
                  INTO _top_ksb_id
                  FROM scheduling.resource__translate__resource_id__to__ksvba__shorthand( _top_resource_id );
                  _top_shifttimes := scheduling.ksvba__get_shifttimes( _top_ksb_id );
                  _top_work_time_day := extract( epoch FROM _top_shifttimes[2] - _top_shifttimes[1] ) / ( 60 * 60 );
                  _top_ksb_ks_ba := ksb_ks_ba FROM ksvba WHERE ksb_id = _top_ksb_id;

                  -- Daten der Kopfkostenstelle übernehmen:
                  _currentSlot.top_resource_id    := _top_resource_id;
                  -- Die Zeitgrenzen des Task-Slots auf der KKS ermitteln. Slot soll für die komplette "An"-Zeit des Tages eingetragen werden.
                  _currentSlot.top_slotStartDate  := date_trunc( 'day', _currentSlot.slotStartDate ) + _top_shifttimes[1];
                  _currentSlot.top_slotEndDate    := date_trunc( 'day', _currentSlot.slotStartDate ) + _top_shifttimes[2];

                  -- Typ der Slots auf der KKS ermitteln.
                  _currentSlot.top_slotType       := _slotType;

                  -- Belastung und Korrekturfaktor
                  _currentSlot.top_usage          := _top_load_to_use;
                  _currentSlot.top_ta_kf          := ( _slotSizeUsed * _factor_tm_ta * _currentSlot.ta_kf ) / ( _top_work_time_day * 60 * 60 );
                  -- Debug: top_ta_kf
                  IF _loglevel >= 4 THEN
                      RAISE NOTICE '%   >> top_ta_kf: % / % = % ', _prefix, round( ( _slotSizeUsed * _factor_tm_ta * _currentSlot.ta_kf / ( 60 * 60 ) ), 4 ), round( _top_work_time_day, 4) , round( _currentSlot.top_ta_kf, 4 );
                  END IF;
              END IF;

              -- Debug: _time_performed
              IF _loglevel >= 5 THEN
                  RAISE NOTICE '%   >> [found selected block] slotSizeUsed: %, slotSizeEffective: %, kf: %, time_performed: %,  time_required: %', _prefix, _slotSizeUsed, ( _slotSizeUsed * _currentSlot.ta_kf ), _currentSlot.ta_kf, _time_performed, _time_required;
              END IF;
              -- Slotliste aktualisieren.
              _currentSet := array_append( _currentSet, _currentSlot  );

              -- Terminierung bis zur ersten Fragmentierung ist gewünscht und Frasgemntierung ist aufgetreten, ...
              IF _allow_fragmentation = 'until' AND _fragmentation_start IS NOT null THEN
                    -- ... dann Slotsuche beenden.
                    IF _loglevel >= 5 THEN
                          RAISE NOTICE '%   >> [end search] due to fragmentation', _prefix;
                    END IF;
                    EXIT timeSlotLoop;
              END IF;

              -- Schleife beenden, wenn wir fertig sind.
              IF ( _time_performed >= _time_required ) THEN
                  EXIT timeSlotLoop;
              END IF;

              -- Nichts zurückgeben, falls wier den vorgegbenen Zeitrahmen überschritten haben.
              IF ( _currentSlot.slotStartDate < _timeframe_start OR _currentSlot.slotEndDate > _timeframe_end ) THEN
                  IF _loglevel >= 4 THEN
                      RAISE NOTICE '%   >> [cancel search] due to exceeding timeframe', _prefix;
                  END IF;
                  _currentSet := array[]::scheduling.resource_timeslotset[];
                  _time_performed := 0;
                  EXIT timeSlotLoop;
              END IF;

          END LOOP;

      -- Maximale Parallelität-Algorithmus
      -- Beschreibung der Terminierungsalgorithmen: https://redmine.prodat-sql.de/projects/terminierung-2021/wiki/Terminierungsalgorithmus#Kopfkostenstellen
      ELSE

          -- Durchlaufe alle Belegungszeiten auf der gegebenen Ressource (Hauptressource) im gegebenen Zeitfenster.
          << timeSlotLoop >>
          FOR _rowHR IN

              SELECT
                  a.ids,
                  a.load,
                  a.date_start,
                  a.date_end,
                  a.types,
                  a.next_start,
                  a.next_end,
                  a.prev_start,
                  a.prev_end,
                  a.next_types,
                  a.next_load,
                  a.prev_types,
                  a.prev_load
              -- Belegungszeiten der Hauptresource
              FROM scheduling.resource_timeline__select(
                        _resource_id        => _resource_id,
                        _target_load        => _load_required,
                        _scenario           => _scenario,
                        _timeframe_start    => _timeframe_start,
                        _timeframe_end      => _timeframe_end,
                        _blocktime_cursor   => _blocktime_refcursor,
                        _checkBlockedTimes  => _checkBlockedTimes,
                        _loglevel           => _loglevel
                  ) a
              ORDER BY
                  CASE WHEN _direction = 'forward' THEN a.date_end END ASC,
                  CASE WHEN _direction <> 'forward' THEN a.date_start END DESC

          LOOP

              IF ( _direction = 'forward' ) THEN

                  IF ( _rowHR.next_start IS NULL ) THEN
                    CONTINUE timeSlotLoop;
                  END IF;

                  _top_timeframe_start := _rowHR.date_end;
                  _top_timeframe_end   := _rowHR.next_start;

              ELSE -- backward

                  IF ( _rowHR.prev_end IS NULL ) THEN
                    CONTINUE timeSlotLoop;
                  END IF;

                  _top_timeframe_start := _rowHR.prev_end;
                  _top_timeframe_end   := _rowHR.date_start;

              END IF;

              -- Nur wenn eine Kopfkostenstelle vorhanden ist.
              IF _top_resource_id IS NOT null THEN
                  -- fetch factor for given timeframes
                  SELECT *
                  INTO _row_factor
                  FROM scheduling.resource_timeline__timeslots_ta_kf__calc(
                      _timeframe_start  => _top_timeframe_start,
                      _timeframe_end    => _top_timeframe_end,
                      _resource_id      => _resource_id
                  );
                  _slotSizeUsed := extract ( epoch FROM _row_factor.slotEndDate - _row_factor.slotStartDate )::numeric * _row_factor.ta_kf / 3600;
                  -- Belastung der Kopfkostenstelle ermittlen.
                  _top_load_to_use :=  scheduling.resource_timeline__timeslots_top_usage__calc(
                                    _top_resource_id  => _top_resource_id
                                  , _date             => CASE WHEN _direction = 'forward' THEN _rowHR.date_end ELSE _rowHR.date_start END::date
                                  , _slotSizeUsed     => ( ( ceil( _slotSizeUsed * _currentSlot.ta_kf ) ) / ( 60 * 60 ) )
                                  , _factor_tm_ta     => _factor_tm_ta
                                  , _loglevel         => _loglevel
                                );
              END IF;

              -- Durchlaufe alle Belegungszeiten auf der ggf. vorhanden zugehörigen Kopfkostenstelle für den aktuellen freien Slot auf
              -- der Hauptressource und extrahiere daraus die zur Abarbeitung freien Zeiten.
              << timeSlotLoopKKS >>
              FOR _rowKKS IN

                  SELECT
                      o.*
                  FROM (
                          -- Keine Kopfkostenstelle vorhanden
                          SELECT
                              a.ids,
                              a.load,
                              a.date_start,
                              a.date_end,
                              a.types,
                              a.next_start,
                              a.next_end,
                              a.prev_start,
                              a.prev_end,
                              a.next_types,
                              a.next_load,
                              a.prev_types,
                              a.prev_load
                          -- Belegungszeiten der Hauptresource
                          FROM ( SELECT _rowHR.ids,
                              _rowHR.load,
                              _rowHR.date_start,
                              _rowHR.date_end,
                              _rowHR.types,
                              _rowHR.next_start,
                              _rowHR.next_end,
                              _rowHR.prev_start,
                              _rowHR.prev_end,
                              _rowHR.next_types,
                              _rowHR.next_load,
                              _rowHR.prev_types,
                              _rowHR.prev_load ) a
                          WHERE _top_resource_id IS NULL

                          UNION

                          -- Kopfkostenstelle vorhanden.
                          SELECT
                              b.ids,
                              b.load,
                              CASE WHEN b.date_start IS NULL THEN NULL ELSE LEAST( b.date_start, c.date_start ) END AS date_start,
                              CASE WHEN b.date_end IS NULL THEN NULL ELSE GREATEST( b.date_end, c.date_end ) END AS date_end,
                              b.types,
                              CASE WHEN b.next_start IS NULL THEN NULL ELSE LEAST( b.next_start, c.next_start ) END AS next_start,
                              CASE WHEN b.next_end IS NULL THEN NULL ELSE GREATEST( b.next_end, c.next_end ) END AS next_end,
                              CASE WHEN b.prev_start IS NULL THEN NULL ELSE LEAST( b.prev_start, c.prev_start ) END AS prev_start,
                              CASE WHEN b.prev_end IS NULL THEN NULL ELSE GREATEST( b.prev_end, c.prev_end ) END AS prev_end,
                              b.next_types,
                              b.next_load,
                              b.prev_types,
                              b.prev_load
                          -- Belegungszeiten der Hauptresource
                          FROM ( SELECT _rowHR.ids,
                              _rowHR.load,
                              _rowHR.date_start,
                              _rowHR.date_end,
                              _rowHR.types,
                              _rowHR.next_start,
                              _rowHR.next_end,
                              _rowHR.prev_start,
                              _rowHR.prev_end,
                              _rowHR.next_types,
                              _rowHR.next_load,
                              _rowHR.prev_types,
                              _rowHR.prev_load ) b
                          -- Belegungszeiten der Kopfkostenstelle
                          JOIN scheduling.resource_timeline__select(
                                    _resource_id        => _top_resource_id,
                                    _target_load        => _top_load_to_use,
                                    _scenario           => _scenario,
                                    _timeframe_start    => _top_timeframe_start,
                                    _timeframe_end      => _top_timeframe_end,
                                    _blocktime_cursor   => _blocktime_refcursor,
                                    _checkBlockedTimes  => _checkBlockedTimes,
                                    _loglevel           => _loglevel
                              ) c ON  (
                                        (
                                              _direction = 'forward'
                                          AND tsrange( b.date_end , b.next_start, '[]' ) && tsrange( c.date_end, c.next_start, '[]' )
                                          AND b.next_start IS NOT NULL
                                          AND c.next_start IS NOT NULL
                                        )
                                    OR
                                        (
                                              _direction <> 'forward'
                                          AND tsrange( b.prev_end, b.date_start, '[]' ) && tsrange( c.prev_end, c.date_start, '[]' )
                                          AND b.prev_end IS NOT NULL
                                          AND c.prev_end IS NOT NULL
                                        )
                              )
                          WHERE _top_resource_id IS NOT NULL
                  ) o
                  ORDER BY
                      CASE WHEN _direction = 'forward' THEN o.date_end END ASC,
                      CASE WHEN _direction <> 'forward' THEN o.date_start END DESC

              LOOP

                  -- checks if the first possible day is the same as start of the timeframe and if so,
                  -- then we check if the time of the timeframe is before the free timeslot, and if not copy it as
                  -- the beginning time of the timeslot
                  IF ( _direction = 'forward' ) THEN

                      _currentSlot.slotStartDate := _rowKKS.date_end;
                      _currentSlot.slotEndDate := _rowKKS.next_start;

                  ELSE -- backward

                      _currentSlot.slotStartDate := _rowKKS.prev_end;
                      _currentSlot.slotEndDate := _rowKKS.date_start;

                  END IF;

                  -- Debugausgabe zu Slot
                  IF _loglevel >= 5 THEN
                      RAISE NOTICE '% slot: %: % ~> %,', _prefix, _direction, _currentSlot.slotStartDate, _currentSlot.slotEndDate;
                  END IF;

                  -- Schon bekannte Felder befüllen.
                  -- Fragemtierung hat (noch) NICHT begonnen oder es handelt sich um Rückwärtsterminierung, ...
                  IF ( _fragmentation_start IS NULL ) OR ( _direction <> 'forward' ) THEN
                      -- ... dann Slot als normalen "Task" festlegen.
                      _currentSlot.slotType := _slotType;
                      _currentSlot.usage    := _load_required;
                  -- Fragemtierung hat (schon) begonnen, ...
                  ELSE
                      -- ... dann Slot als Blockierung festlegen.
                      _currentSlot.slotType := 'task.blocktime';
                      _currentSlot.usage    := ifthen( _load_required = 0.0, 1.0, _load_required );
                  END IF;

                  _useableSlotSize := ceil( extract ( epoch FROM _currentSlot.slotEndDate - _currentSlot.slotStartDate )::numeric); -- Wir wollen nur auf ganze Sekunden aufgerundete Intervalgrößen, da es ansosnsten zu komischen Rundungseffekten kommen kann. Daher hier die Verwendung der Funktion "ceil".

                  -- fetch factor for given timeframes
                  SELECT *
                  INTO _row_factor
                  FROM scheduling.resource_timeline__timeslots_ta_kf__calc(
                      _timeframe_start  => _currentSlot.slotStartDate,
                      _timeframe_end    => _currentSlot.slotEndDate,
                      _resource_id      => _resource_id
                  );

                  _currentSlot.slotStartDate := _row_factor.slotStartDate;
                  _currentSlot.slotEndDate := _row_factor.slotEndDate;
                  _currentSlot.ta_kf := _row_factor.ta_kf;

                  _useableSlotSize := ceil( extract( epoch FROM _row_factor.slotEndDate - _row_factor.slotStartDate )::numeric ); -- Wir wollen nur auf ganze Sekunden aufgerundete Intervalgrößen, da es ansosnsten zu komischen Rundungseffekten kommen kann. Daher hier die Verwendung der Funktion "ceil".

                  IF _useableSlotSize > 0 AND _row_factor.ta_kf <> 0 THEN

                      -- Wenn die geplante Arbeitszeit größer ist als der freie Zeitslot, dann belege den vollen freien Zeitslot.
                      IF ( _time_performed + ceil( _useableSlotSize * _row_factor.ta_kf ) <= _time_required ) THEN

                          _slotSizeUsed := _useableSlotSize;

                      -- Die gelpante bzw. benötigte Arbeitszeit passt in den freien Zeitslot.
                      ELSE
                          _slotSizeUsed := ceil( ( _time_required - _time_performed ) / _row_factor.ta_kf ); -- Wir wollen nur auf ganze Sekunden aufgerundete Intervalgrößen, da es ansosnsten zu komischen Rundungseffekten kommen kann. Daher hier die Verwendung der Funktion "ceil".

                          IF _direction = 'forward' THEN
                              _currentSlot.slotEndDate := _row_factor.slotStartDate + _slotSizeUsed * interval '1s';
                          ELSE
                              _currentSlot.slotStartDate := _row_factor.slotEndDate - _slotSizeUsed * interval '1s';
                          END IF;

                      END IF;

                      -- Kopfkostenstelle existiert und Fragmentierung hat (noch) NICHT begonnen oder Fragmentierungsrichtung ist rückwärts, ...
                      IF ( _top_resource_id IS NOT null ) AND ( ( _currentSlot.slotStartDate < coalesce( _fragmentation_start, 'infinity'::timestamp ) ) OR ( _direction <> 'forward' ) ) THEN
                          -- ... dann Daten der Kopfkostenstelle übernehmen
                          _currentSlot.top_resource_id    := _top_resource_id;
                          _currentSlot.top_slotStartDate  := _currentSlot.slotStartDate;
                          _currentSlot.top_slotEndDate    := _currentSlot.slotEndDate;
                          _currentSlot.top_slotType       := _slotType;
                          _currentSlot.top_ta_kf          := _currentSlot.ta_kf;
                          _currentSlot.top_usage          := _top_load_to_use;
                      END IF;

                      _time_performed := _time_performed + ceil( _useableSlotSize * _row_factor.ta_kf );
                      -- Debug: _time_performed
                      IF _loglevel >= 4 THEN
                          RAISE NOTICE '%   >> [found selected block] slotSizeUsed: %, slotSizeEffective: %, kf: %, time_performed: %,  time_required: %', _prefix, _slotSizeUsed, ( _slotSizeUsed * _currentSlot.ta_kf ), _currentSlot.ta_kf, _time_performed, _time_required;
                      END IF;

                      _currentSet := array_append( _currentSet, _currentSlot  );

                      -- quit the loop - we have what we need
                      IF ( _time_performed >= _time_required ) THEN
                          EXIT timeSlotLoop;
                      END IF;

                  END IF;

                  -- Fragmentierung ist verboten oder hat noch nicht gestartet oder Terminierungsrichtung ist rückwärts, ...
                  -- Bei erlaubter Fragmentierung und Terminierungsrichtung rückwärts IMMER auf Fragmentierung prüfen, auch wenn schon Fragmentierung erkannt wurde, um letztendlich den frühesten Fragmentierungszeitpunkt zu finden, aber nur, wenn vorher schon ein freier Slot belegt wurde.
                  IF ( ( _allow_fragmentation <> 'yes' ) OR ( _fragmentation_start IS NULL ) OR ( _direction <> 'forward' ) ) AND ( _time_performed > 0 ) THEN
                      -- ... dann auf Fragmentierung prüfen:
                      -- Debugausgabe zur Prüfung auf Fragmentierung
                      IF _loglevel >= 6 THEN
                          IF _direction = 'forward' THEN
                              RAISE NOTICE '%   >> [check for fragmentation] next_types: %, next_load: %, load_required: %;', _prefix, _rowHR.next_types, _rowHR.next_load, _load_required;
                          ELSE -- backward
                              RAISE NOTICE '%   >> [check for fragmentation] prev_types: %, prev_load: %, load_required: %;', _prefix, _rowHR.prev_types, _rowHR.prev_load, _load_required;
                          END IF;
                      END IF;
                      -- Prüfung ob die nächste Belegung eine Abarbeitung eines anderen AG ("task") ist.
                      -- Andere Belegungsgründe als "task", "task.blocktype" bzw. 'task.buffer' zählen nicht als Fragmentierung.
                      -- Ein Task-Block zählt nur, wenn die gesamtmögliche Belegung durch den hinzukommenden aktuellen AG überschritten werden würde.
                      -- Bei den Kopfkostenstellen darf es Fragmentierung geben.
                      IF (
                            -- Vorwärtsterminierung, Hauptressource
                            (
                                  _direction = 'forward'
                              AND (
                                        _rowKKS.next_types && array[ 'task', 'task.blocktime' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task'
                                    OR _rowKKS.next_types && array[ 'task', 'task.blocktime', 'task.buffer' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task.buffer'
                                  )
                              AND _rowKKS.next_load + _load_required > 1
                            )
                          OR
                            -- Rückwärtsterminierung, Hauptressource
                            (
                                  _direction <> 'forward'
                              AND (
                                        _rowKKS.prev_types && array[ 'task', 'task.blocktime' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task'
                                    OR _rowKKS.prev_types && array[ 'task', 'task.blocktime', 'task.buffer' ]::scheduling.resource_timeline_blocktype[] AND _slotType = 'task.buffer'
                                  )
                              AND _rowKKS.prev_load + _load_required > 1
                            )
                      ) THEN
                          -- Fragmentierung ist verboten, ...
                          IF ( _allow_fragmentation = 'no' ) THEN
                              -- ... dann wird das Ergebnis der aktuellen Suche nach freien Blöcken verworfen.
                              -- Es wird ab dem nächsten Belegungsblock wieder nach freien Lücken gesucht.
                              IF _loglevel >= 5 THEN
                                  RAISE NOTICE '%   >> [reset search] due to fragmentation', _prefix;
                              END IF;

                              _time_performed := 0;
                              _currentSet := array[]::scheduling.resource_timeslotset[];
                          -- Fragmentierung ist erlaubt, ...
                          ELSE
                              -- ... dann merken, das ab nächstem Block die Fragmentierung begonnen hat, aber nur, wenn noch keine Fragmentierung gefunden wurde.
                              IF ( _fragmentation_start IS null ) OR ( _direction <> 'forward' ) THEN
                                  IF _direction = 'forward' THEN
                                      _fragmentation_start := _currentSlot.slotEndDate;
                                  ELSE
                                      _fragmentation_start := _currentSlot.slotstartDate;
                                  END IF;
                                  IF _loglevel >= 5 THEN
                                      RAISE NOTICE '%   >> [fragmentation found] fragmentation_start: %', _prefix, _fragmentation_start;
                                  END IF;
                              END IF;
                          END IF;
                      END IF;
                  END IF;

                  -- skip size zero slots
                  IF ( _useableSlotSize = 0 ) THEN
                      -- we cannot skip zero width slots, because we need to detect
                      -- tasks causing fragmentation
                      -- TODO AXS: Merkwürdig. Das hier ist ja ein "frei"-Slot, in dem eine Aufgabe einer Ressource einsortiert werden soll. Der ist unbenutzbar, wenn Länge 0.
                      -- [x] AXS: Außerdem wird in wenigen Zeilen die Schleife so und so verlassen, wenn Slotlänge 0.
                      --> Das nächste CONTINUE springt nur aus dem _row_factor-LOOP.
                      IF _loglevel >= 5 THEN
                          RAISE NOTICE '%   >> [skip block] due to zero slot size', _prefix;
                      END IF;
                      -- CONTINUE timeSlotLoop;
                  END IF;

                  -- Terminierung bis zur ersten Fragmentierung ist gewünscht und Frasgemntierung ist aufgetreten, ...
                  IF _allow_fragmentation = 'until' AND _fragmentation_start IS NOT null THEN
                        -- ... dann Slotsuche beenden.
                        IF _loglevel >= 5 THEN
                              RAISE NOTICE '%   >> [end search] due to fragmentation', _prefix;
                        END IF;
                        EXIT timeSlotLoop;
                  END IF;

                  -- return nothing if we exceed our timeframe
                  IF ( _currentSlot.slotStartDate < _timeframe_start OR _currentSlot.slotEndDate > _timeframe_end ) THEN

                      IF _loglevel >= 4 THEN
                          RAISE NOTICE '%   >> [cancel search] due to exceeding timeframe', _prefix;
                      END IF;

                      _currentSet := array[]::scheduling.resource_timeslotset[];
                      _time_performed := 0;

                      EXIT timeSlotLoop;

                  END IF;

              END LOOP;

          END LOOP;

      END IF;

      -- Unvollständige Ergebnisse zurücksetzen. Außer wenn Terminierung bis zur ersten Frasgmentierung gewünscht ist und Fragmentierung aufgetreten ist, dann sind unvollständige Ergbenisse erlaubt.
      IF ( _time_performed < _time_required ) AND NOT( _allow_fragmentation = 'until' AND _fragmentation_start IS NOT null ) THEN

          _time_performed := 0;
          _currentSet := array[]::scheduling.resource_timeslotset[];

          IF _loglevel >= 4 THEN
              RAISE NOTICE '% nothing found', _prefix;
          END IF;
      END IF;

      -- Es wurde rückwärts terminiert und Fragmentierung war erlaubt und ist aufgetreten, dann ...
      IF ( _direction <> 'forward' ) AND ( _fragmentation_start IS NOT null ) THEN -- "_fragmentation_start" kann nur gesetzt werden, wenn Fragmentierung erlaubt.
          -- ... für die Fragmentierten Blöcke den Typ auf Blockierung stellen und die zugehörigen Blöcke auf der Kopfkostenstelle löschen.
          FOR i IN array_lower(_currentSet, 1)..array_upper(_currentSet, 1) LOOP
              IF _currentSet[i].slotStartDate >= coalesce( _fragmentation_start, 'infinity'::timestamp ) THEN
                  _currentSlot := _currentSet[i];
                  -- _currentSet[i].slotType         := 'task.blocktime'; -- TODO AXS: Erzeugt einen Syntax error. Herausfinden, wie korrekte Notation beim Beschreiben.
                  _currentSlot.slotType           := 'task.blocktime';
                  _currentSlot.usage              := ifthen( _load_required = 0.0, 1.0, _load_required );
                  _currentSlot.top_resource_id    := null;
                  _currentSlot.top_slotStartDate  := null;
                  _currentSlot.top_slotEndDate    := null;
                  _currentSlot.top_slotType       := null;
                  _currentSlot.top_usage          := null;
                  _currentSlot.top_ta_kf          := null;
                  _currentSet[i] := _currentSlot;
              END IF;
          END LOOP;
      END IF;

      -- return data found
      FOREACH _rowHR IN ARRAY _currentSet
      LOOP
          RETURN QUERY SELECT tsystem.timestamp__floor_to_interval( _rowHR.slotStartDate, _time_grid ), tsystem.timestamp__ceil_to_interval( _rowHR.slotEndDate, _time_grid ), _rowHR.slotType, _rowHR.ta_kf, _rowHR.usage, _rowHR.top_resource_id, tsystem.timestamp__floor_to_interval( _rowHR.top_slotStartDate, _time_grid ), tsystem.timestamp__ceil_to_interval( _rowHR.top_slotEndDate, _time_grid ), _rowHR.top_slotType, _rowHR.top_ta_kf, _rowHR.top_usage;
      END LOOP;

  END $$ language plpgsql;
